/*----------------------------------------------------------------------------*/
/* ARBRES BINAIRES                                                    ARBRE.C */
/*                                                         Par Bruno Bachelet */
/*----------------------------------------------------------------------------*/
/* Copyright (c) 1999-2016
   Bruno Bachelet - bruno@nawouak.net - http://www.nawouak.net

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the license, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
   more details (http://www.gnu.org).
*/

/* Headers -------------------------------------------------------------------*/
#include "arbre.h"

/* Variables globales --------------------------------------------------------*/
BOOLEEN EQUILIBRAGE = FAUX;

/* 1 - InitialiserArbre ------------------------------------------------------*/

void initialiserArbre(ARBRE * a) { *a = NULL; }

/* 2 - PreparerNoeud ---------------------------------------------------------*/

NOEUD * preparerNoeud(ELEMENT e) {
 NOEUD * n = ALLOUER(NOEUD,1);

 if (n != NULL) {
  n->elt = e;
  n->fg = NULL;
  n->fd = NULL;
  n->h = 0;
 }

 return n;
}

/* 3 - AjouterNoeud ----------------------------------------------------------*/

BOOLEEN ajouterNoeud(ARBRE * a, NOEUD * n) {
 BOOLEEN ok;

 /* Sans reequilibrage. */
 if (!EQUILIBRAGE) {
  if (*a == NULL) {
   *a = n;
   return VRAI;
  }

  if (cle(n->elt) == cle((*a)->elt)) return FAUX;
  if (cle(n->elt) < cle((*a)->elt)) return ajouterNoeud(&((*a)->fg),n);
  return ajouterNoeud(&((*a)->fd),n);
 }

 /* Avec reequilibrage. */
 else {
  if (*a == NULL) {
   *a = n;
   n->h = 1;
   return VRAI;
  }

  if (cle(n->elt) == cle((*a)->elt)) return FAUX;
  if (cle(n->elt) < cle((*a)->elt)) ok = ajouterNoeud(&((*a)->fg),n);
  else ok = ajouterNoeud(&((*a)->fd),n);

  if (ok) {
   majHauteur(*a);
   reequilibrer(a);
  }

  return ok;
 }
}

/* 4 - AjouterElement --------------------------------------------------------*/

BOOLEEN ajouterElement(ARBRE * a, ELEMENT e) {
 NOEUD * n = preparerNoeud(e);

 if (n == NULL) return FAUX;

 if (!ajouterNoeud(a,n)) {
  LIBERER(n);
  return FAUX;
 }

 return VRAI;
}

/* 5 - ExisteCle -------------------------------------------------------------*/

BOOLEEN existeCle(ARBRE a, CLE c) {
 if (a == NULL) return FAUX;
 if (c == cle(a->elt)) return VRAI;
 if (c < cle(a->elt)) return existeCle(a->fg,c);
 return existeCle(a->fd,c);
}

/* 6 - ExtraireMaximum -------------------------------------------------------*/

NOEUD * extraireMaximum(ARBRE * a) {
 NOEUD * n;

 /* Sans reequilibrage. */
 if (!EQUILIBRAGE) {
  if (*a == NULL) return NULL;

  if ((*a)->fd == NULL) {
   n = *a;
   *a = (*a)->fg;
   return n;
  }

  return extraireMaximum(&((*a)->fd));
 }

 /* Avec reequilibrage. */
 else {
  if (*a == NULL) return NULL;

  if ((*a)->fd == NULL) {
   n = *a;
   *a = (*a)->fg;
   return n;
  }

  n = extraireMaximum(&((*a)->fd));

  if (n != NULL) {
   majHauteur(*a);
   reequilibrer(a);
  }

  return n;
 }
}

/* 7 - SupprimerRacine -------------------------------------------------------*/

BOOLEEN supprimerRacine(ARBRE * a) {
 NOEUD * n;

 if (*a == NULL) return FAUX;
 n = *a;

 if (n->fg == NULL) *a = n->fd;
 else if (n->fd == NULL) *a = n->fg;
 else {
  *a = extraireMaximum(&(n->fg));
  (*a)->fg = n->fg;
  (*a)->fd = n->fd;
 }

 LIBERER(n);
 return VRAI;
}

/* 8 - ExtraireElement -------------------------------------------------------*/

BOOLEEN extraireElement(ARBRE * a, CLE c, ELEMENT * e) {
 BOOLEEN ok;

 /* Sans reequilibrage. */
 if (!EQUILIBRAGE) {
  if (*a == NULL) return FAUX;

  if (c < cle((*a)->elt)) return extraireElement(&((*a)->fg),c,e);
  if (c > cle((*a)->elt)) return extraireElement(&((*a)->fd),c,e);

  *e = (*a)->elt;
  return supprimerRacine(a);
 }

 /* Avec reequilibrage. */
 else {
  if (*a == NULL) return FAUX;

  if (c < cle((*a)->elt)) ok = extraireElement(&((*a)->fg),c,e);
  else if (c > cle((*a)->elt)) ok = extraireElement(&((*a)->fd),c,e);
  else {
   *e = (*a)->elt;
   ok = supprimerRacine(a);
  }

  if (ok && *a != NULL) {
   majHauteur(*a);
   reequilibrer(a);
  }

  return ok;
 }
}

/* 9 - MajHauteur ------------------------------------------------------------*/

void majHauteur(NOEUD * n) {
 int hd;
 int hg;

 if (n->fg == NULL) hg = 0;
 else hg = n->fg->h;

 if (n->fd == NULL) hd = 0;
 else hd = n->fd->h;

 n->h = MAX(hd,hg) + 1;
}

/* 10 - Desequilibre ---------------------------------------------------------*/

int desequilibre(ARBRE a) {
 int hd;
 int hg;

 if (a == NULL) return 0;

 if (a->fg == NULL) hg = 0;
 else hg = a->fg->h;

 if (a->fd == NULL) hd = 0;
 else hd = a->fd->h;

 return (hg - hd);
}

/* 11 - RotationRD -----------------------------------------------------------*/

BOOLEEN rotationRD(ARBRE * a) {
 NOEUD * n = *a;

 if (n->fg == NULL) {
  printf("Rotation impossible.\n");
  return FAUX;
 }

 *a = n->fg;
 n->fg = (*a)->fd;
 (*a)->fd = n;

 if (EQUILIBRAGE) {
  majHauteur(n);
  majHauteur(*a);
 }

 return VRAI;
}

/* 12 - RotationRG -----------------------------------------------------------*/

BOOLEEN rotationRG(ARBRE * a) {
 NOEUD * n = *a;

 if (n->fd == NULL) {
  printf("Rotation impossible.\n");
  return FAUX;
 }

 *a = n->fd;
 n->fd = (*a)->fg;
 (*a)->fg = n;

 if (EQUILIBRAGE) {
  majHauteur(n);
  majHauteur(*a);
 }

 return VRAI;
}

/* 13 - RotationRGD ----------------------------------------------------------*/

BOOLEEN rotationRGD(ARBRE * a) {
 if ((*a)->fg == NULL) {
  printf("Rotation impossible.\n");
  return FAUX;
 }

 if (!rotationRG(&((*a)->fg))) return FAUX;
 return rotationRD(a);
}

/* 14 - RotationRDG ----------------------------------------------------------*/

BOOLEEN rotationRDG(ARBRE * a) {
 if ((*a)->fd == NULL) {
  printf("Rotation impossible.\n");
  return FAUX;
 }

 if (!rotationRD(&((*a)->fd))) return FAUX;
 return rotationRG(a);
}

/* 15 - Reequilibrer ---------------------------------------------------------*/

void reequilibrer(ARBRE * a) {
 int d = desequilibre(*a);

 if (d == +2) {
  if (desequilibre((*a)->fg) == -1) rotationRGD(a);
  else rotationRD(a);
 }
 else if (d == -2) {
  if (desequilibre((*a)->fd) == +1) rotationRDG(a);
  else rotationRG(a);
 }
}

/* Fin -----------------------------------------------------------------------*/
